/*********************************************************************/
// webdar - a web server and interface program to libdar
// Copyright (C) 2013-2025 Denis Corbin
//
// This file is part of Webdar
//
//  Webdar is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  Webdar is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with Webdar.  If not, see <http://www.gnu.org/licenses/>
//
//----
//  to contact the author: dar.linux@free.fr
/*********************************************************************/

    // C system header files
#include "my_config.h"

extern "C"
{

}

    // C++ system header files


    // webdar headers
#include "html_text.hpp"
#include "tokens.hpp"
#include "exceptions.hpp"
#include "webdar_css_style.hpp"
#include "webdar_tools.hpp"

    //
#include "html_menu.hpp"

using namespace std;

const string html_menu::changed = "html_menu_changed";

html_menu::html_menu()
{
    current_mode = 0;
    previous_mode = 0;

    register_name(changed); // add the "html_menu_changed" event to this object
}

html_menu::~html_menu()
{
    item.clear();
}

void html_menu::add_entry(const string & label, const string & tag)
{
    unsigned int num = item.size();
    string event_name = webdar_tools_convert_to_string(num);

    item.push_back(cell(label, event_name, tag));
    if(item.size() != num + 1)
	throw WEBDAR_BUG;

    item.back().ibtn->record_actor_on_event(this, event_name);

    if(label != "")
	webdar_css_style::normal_button(*(item.back().ibtn));
    else
    {
	webdar_css_style::void_button(*(item.back().ibtn));
	item.back().ibtn->set_enabled(false);
    }

    item.back().ibtn->add_css_class(webdar_css_style::spacing_vertical);

	/// building the body_builder tree
    adopt(item.back().ibtn);

    if(item.size() == 1)
	set_current_mode(0, true);
}

void html_menu::set_current_mode(unsigned int mode, bool force)
{
    unsigned int size = item.size();
    bool has_changed = (mode != current_mode) || force;

    if(mode >= size)
	throw WEBDAR_BUG;

    if(current_mode >= size)
    	throw WEBDAR_BUG;

    if(item[current_mode].ibtn == nullptr)
	throw WEBDAR_BUG;

    if(item[mode].ibtn == nullptr)
	throw WEBDAR_BUG;

	/// all is fine, we can go on modifying the objects

    if(has_changed)
    {
	if(item[current_mode].ibtn->get_label() == "")
	    webdar_css_style::void_button(*(item[current_mode].ibtn));
	else
	    webdar_css_style::normal_button(*(item[current_mode].ibtn));
	item[current_mode].ibtn->add_css_class(webdar_css_style::spacing_vertical);

	webdar_css_style::active_button(*(item[mode].ibtn));
	item[mode].ibtn->add_css_class(webdar_css_style::spacing_vertical);

	previous_mode = current_mode;
	current_mode = mode;
	act(changed); // trigger the "changed" event
	my_body_part_has_changed();
    }
}

string html_menu::get_current_tag() const
{
    if(current_mode >= item.size())
	throw WEBDAR_BUG;

    return item[current_mode].itag;
}

void html_menu::set_current_tag(const string & tag)
{
    unsigned int i = 0;

    while(i < item.size() && item[i].itag != tag)
	++i;

    if(i < item.size())
	set_current_mode(i);
    else
	throw WEBDAR_BUG; // unknown tag in this html_menu
}

string html_menu::get_previous_tag() const
{
    if(previous_mode >= item.size())
	throw WEBDAR_BUG;

    return item[previous_mode].itag;
}

string html_menu::get_current_label() const
{
    if(current_mode >= item.size())
	throw WEBDAR_BUG;

    if(item[current_mode].ibtn == nullptr)
	throw WEBDAR_BUG;

    return item[current_mode].ibtn->get_label();
}

void html_menu::on_event(const string & event_name)
{
	// we have registered on the event generated by each html_button
	// the event_name is the string of the decimal index of the button

    int num;

    try
    {
	num = webdar_tools_convert_to_int(event_name);
    }
    catch(exception_range & e)
    {
	throw WEBDAR_BUG;
	    // received an event we don't expect
    }

    set_current_mode(num);
	// set_current_mode() will eventually trigger my_body_part_has_changed();
}

string html_menu::inherited_get_body_part(const chemin & path,
					  const request & req)
{
	// reading the requested path to determin
	// whether a change of mode is required
	// before starting providing an response
	// from any component or subcomponent

    chemin target = req.get_uri().get_path();

    if(target.size() > 2)
    {
	target.pop_back();
	target.pop_back();

	if(target == get_path())
	{
		// the requested link is us

	    unsigned int i = 0;
	    unsigned int size = item.size();
	    string target_s = req.get_uri().get_path().display();

		// looking which button index it is:

	    while(i < size && item[i].ibtn != nullptr && item[i].ibtn->get_url() != target_s)
		++i;
	    if(i < size)
		set_current_mode(i);
	}
    }

    return html_div::inherited_get_body_part(path, req);
}

void html_menu::new_css_library_available()
{

	// recording those classes and selectors to the css_library

    unique_ptr<css_library> & csslib = lookup_css_library();
    if(!csslib)
	throw WEBDAR_BUG;

    webdar_css_style::update_library(*csslib);
}
