// re2swift $INPUT -o $OUTPUT --api generic --tags
import Foundation

struct SemVer: Equatable { var major: Int, minor: Int, patch: Int }

struct Input {
  static let bufferSize = 4095
  static let tagNone = -1

  var yyinput  = ContiguousArray<UInt8>(repeating: 0, count: Self.bufferSize + 1)
  var yylimit  = Self.bufferSize
  var yycursor = Self.bufferSize
  var yymarker = Self.bufferSize
  var token    = Self.bufferSize
  // Intermediate tag variables must be part of the lexer state passed to YYFILL.
  // They don't correspond to tags and should be autogenerated by re2c.
  /*!stags:re2c format = "  var @@ = Self.tagNone\n"; */
  var eof = false

  let file: FileHandle
}

extension Input {
  mutating func lex() -> [SemVer]? {
    var semVers = [SemVer]()
    semVers.reserveCapacity(Self.bufferSize)

    // Final tag variables available in semantic action.
    /*!svars:re2c format = "    var @@: Int\n"; */

    parse: while true {
      self.token = self.yycursor
      /*!re2c
        re2c:api  = record;
        re2c:eof  = 0;
        re2c:tags = 1;
        re2c:yyrecord = "self";
        re2c:YYFILL   = "self.fill() == .ok";

        num = [0-9]+;

        num @t1 "." @t2 num @t3 ("." @t4 num)? [\n] {
          semVers.append(SemVer(
            major: self.s2n(self.token..<t1),
            minor: self.s2n(t2..<t3),
            patch: t4 != Self.tagNone ? self.s2n(t4..<(self.yycursor - 1)) : 0
          ))
          continue parse
        }
        $ { return semVers }
        * { return nil }
      */
    }
  }

  func s2n(_ range: Range<Int>) -> Int {
    self.yyinput[range].reduce(0) { accum, digit in
      accum * 10 + Int(digit - UInt8(ascii: "0"))
    }
  }

  mutating func fill() -> FillStatus {
    guard !self.eof else {
      return .eof
    }

    let shift = self.token
    let used = self.yylimit - self.token
    let free = Self.bufferSize - used

    // Error: Lexeme too long. In the real world we could reallocate a larger buffer.
    guard self.token >= 1 else {
      return .longLexeme
    }

    // Shift buffer contents, discarding everything up to the current token.
    self.yyinput.replaceSubrange(..<used, with: self.yyinput[shift..<self.yylimit])
    self.yylimit   -= shift
    self.yycursor  -= shift
    self.yymarker &-= shift  // May underflow is marker is unused
    self.token = 0
    // Tag variables need to be shifted like other input positions. The check
    // for `tagNone` is only needed if some tags are nested inside of alternative
    // or repetition, so that they can have `tagNone` value.
    /*!stags:re2c format = "    if self.@@ != Self.tagNone { self.@@ -= shift }\n"; */

    // Fill free space at the end of the buffer with new data from file.
    do {
      if let data = try self.file.read(upToCount: free) {
        self.yyinput.replaceSubrange(self.yylimit..<(self.yylimit + data.count), with: data)
        self.yylimit += data.count
      }
    } catch {
      fatalError("cannot read from file: \(error.localizedDescription)")
    }
    self.yyinput[self.yylimit] = 0  // append sentinel
    self.eof = self.yylimit < Self.bufferSize

    return .ok
  }

  enum FillStatus {
    case ok, eof, longLexeme
  }
}

extension SemVer: CustomStringConvertible {
  var description: String { "\(self.major).\(self.minor).\(self.patch)" }
}

let fileName = "input"
let semVer = SemVer(major: 1, minor: 22, patch: 333)
let expect = [SemVer](repeating: semVer, count: Input.bufferSize)

// Prepare input file (make sure it exceeds buffer size).
guard FileManager.default.createFile(
    atPath: fileName,
    contents: Data(String(repeating: "\(semVer)\n", count: Input.bufferSize).utf8)
) else {
  fatalError("failed to write file \"\(fileName)\"")
}

// Reopen input file for reading.
guard let file = FileHandle(forReadingAtPath: fileName) else {
  throw NSError(domain: NSCocoaErrorDomain, code: CocoaError.fileReadNoSuchFile.rawValue)
}

// Initialize lexer state. Buffer is set to zero, triggering YYFILL.
var `in` = Input(file: file)

// Run the lexer and check the results.
guard let actual = `in`.lex() else {
  fatalError("parser error")
}

assert(actual == expect)

// Cleanup: remove input file.
try file.close()
try FileManager.default.removeItem(atPath: fileName)
